// ==================== НАСТРОЙКИ ====================
const SERVER_URL = "https://id-tunnel-b.duckdns.org";
const SITE_NAME = window.location.hostname.replace(/^www\./, "");

// ДИНАМИЧЕСКИЕ СЕЛЕКТОРЫ
let DYNAMIC_SELECTORS = '';

// Кэш настроек сайта и групп
let siteSettings = {
    minDuplicates: 2,
    minDate: "2025-09-22",
    useMinDuplicates: true,
    useMinDate: true,
    siteGroups: []
};

// Кэш парсеров
let siteParsers = [];
let lastParsersLoad = 0;
const PARSERS_CACHE_TIME = 60000; // 1 минута

// КЭШ ДЛЯ ЗАПИСЕЙ И СОСТОЯНИЕ
const recordsCache = new Map();
const updateInProgress = new Set();
let lastSettingsLoad = 0;
const SETTINGS_CACHE_TIME = 30000; // 30 секунд

// ==================== ФУНКЦИИ СЕЛЕКТОРОВ ====================

// Функция для обновления селекторов из парсеров
function updateDynamicSelectors() {
    if (!siteParsers || siteParsers.length === 0) {
        console.log('ℹ️ Нет парсеров для обновления селекторов');
        DYNAMIC_SELECTORS = ''; // Очищаем селекторы если нет парсеров
        return;
    }
    
    const allSelectors = new Set();
    
    // ДОБАВЛЯЕМ ТОЛЬКО СЕЛЕКТОРЫ ИЗ ПАРСЕРОВ - без базовых
    siteParsers.forEach(parser => {
        if (parser.selectors && Array.isArray(parser.selectors)) {
            parser.selectors.forEach(selector => {
                if (selector && selector.trim() && selector.length > 2) {
                    allSelectors.add(selector.trim());
                }
            });
        }
    });
    
    // Обновляем глобальную переменную
    DYNAMIC_SELECTORS = Array.from(allSelectors).join(', ');
    
    console.log(`🎯 Обновлены динамические селекторы из парсеров: ${allSelectors.size} селекторов`, Array.from(allSelectors));
    
    // Уведомляем об обновлении
    chrome.runtime.sendMessage({
        action: 'selectorsUpdated',
        selectorsCount: allSelectors.size,
        selectors: Array.from(allSelectors).slice(0, 10)
    });
}

// Функция для получения текущих селекторов
function getCurrentSelectors() {
    return DYNAMIC_SELECTORS;
}

// Функция проверки соответствия элемента селекторам парсеров
function elementMatchesAnyParserSelector(element) {
    if (!siteParsers || siteParsers.length === 0) {
        return true; // Если парсеров нет, пропускаем все элементы
    }
    
    for (const parser of siteParsers) {
        if (parser.selectors && Array.isArray(parser.selectors)) {
            for (const selector of parser.selectors) {
                try {
                    if (selector && element.matches && element.matches(selector)) {
                        console.log(`🎯 Элемент соответствует селектору парсера "${parser.name}": ${selector}`);
                        return true;
                    }
                } catch (selectorError) {
                    console.warn(`❌ Ошибка в селекторе "${selector}" парсера "${parser.name}":`, selectorError);
                }
            }
        }
    }
    
    return false;
}

// ==================== ЗАГРУЗКА ПАРСЕРОВ ====================

async function loadSiteParsers() {
    try {
        console.log(`🔍 Загрузка парсеров ТОЛЬКО для текущего сайта: ${SITE_NAME}`);
        
        // ЗАГРУЖАЕМ ПАРСЕРЫ ТОЛЬКО ДЛЯ ТЕКУЩЕГО САЙТА
        const res = await fetch(`${SERVER_URL}/api/parsers?site=${encodeURIComponent(SITE_NAME)}`);
        if (res.ok) {
            const parsers = await res.json();
            // Фильтруем только активные парсеры
            siteParsers = parsers.filter(parser => parser.is_active !== false);
            console.log(`✅ Загружено ${siteParsers.length} парсеров для сайта ${SITE_NAME}`, siteParsers);
        } else {
            console.warn(`❌ Ошибка загрузки парсеров для ${SITE_NAME}:`, res.status);
            siteParsers = [];
        }
        
        // ОБНОВЛЯЕМ СЕЛЕКТОРЫ после загрузки парсеров
        updateDynamicSelectors();
        
        // Пересканируем страницу с новыми селекторами
        setTimeout(() => {
            rescanPageWithNewSelectors();
        }, 500);
        
        // Уведомляем popup
        chrome.runtime.sendMessage({
            action: 'parsersUpdated',
            parsers: siteParsers,
            parsersCount: siteParsers.length,
            selectorsCount: Array.from(new Set(siteParsers.flatMap(p => p.selectors || []))).length,
            site: SITE_NAME,
            groups: siteSettings.siteGroups || []
        });
        
    } catch (e) {
        console.warn("❌ Ошибка загрузки парсеров:", e);
        siteParsers = [];
    }
}
// ==================== ОПТИМИЗИРОВАННАЯ ЗАГРУЗКА НАСТРОЕК ====================

async function loadSiteSettings() {
    const now = Date.now();
    if (now - lastSettingsLoad < SETTINGS_CACHE_TIME) {
        return;
    }
    
    try {
        const res = await fetch(`${SERVER_URL}/api/site-settings?site=${encodeURIComponent(SITE_NAME)}`);
        
        if (res.ok) {
            const settings = await res.json();
            
            siteSettings = {
                minDuplicates: settings.minDuplicates == null ? 2 : Number(settings.minDuplicates),
                minDate: settings.minDate || getDefaultMinDate(),
                useMinDuplicates: settings.useMinDuplicates === false ? false : true,
                useMinDate: settings.useMinDate === false ? false : true,
                siteGroups: settings.siteGroups || []
            };
            
            chrome.storage.local.set({ currentSiteSettings: siteSettings });
            lastSettingsLoad = now;
            
            // Уведомляем popup об обновлении
            chrome.runtime.sendMessage({
                action: 'settingsUpdated',
                settings: siteSettings,
                site: SITE_NAME
            });
        }
    } catch (e) {
        console.warn("❌ Ошибка загрузки настроек:", e);
    }
}

function getDefaultMinDate() {
    const date = new Date();
    date.setDate(date.getDate() - 30);
    return date.toISOString().split('T')[0];
}

// ==================== ПАРСЕРЫ ID ====================

function extractIDFromElement(el) {
    if (!el || !el.nodeType) {
        return null;
    }
    
    // ИСПОЛЬЗУЕМ ТОЛЬКО ПАРСЕРЫ - никаких стандартных методов
    if (siteParsers.length > 0) {
        const id = extractIDWithParsers(el);
        if (id && isValidId(id)) {
            console.log(`🎯 Парсер извлёк ВАЛИДНЫЙ ID: ${id}`);
            return id;
        }
    }
    
    return null;
}

// Улучшенная функция парсеров
function extractIDWithParsers(element) {
    const html = element.outerHTML;
    let foundId = null;
    
    if (!siteParsers || siteParsers.length === 0) {
        return null;
    }
    
    for (const parser of siteParsers) {
        if (foundId) break;
        
        try {
            console.log(`🔍 Применяем парсер "${parser.name}" для сайта ${parser.site}`);
            
            if (parser.pattern && parser.pattern.length > 2) {
                try {
                    const regex = new RegExp(parser.pattern, 'gi');
                    regex.lastIndex = 0; // Сбрасываем позицию
                    
                    const match = regex.exec(html);
                    if (match) {
                        // Используем первую группу захвата или всё совпадение
                        foundId = match[1] || match[0];
                        if (foundId) {
                            foundId = foundId.trim();
                            console.log(`✅ Парсер "${parser.name}" извлёк: ${foundId}`);
                            break;
                        }
                    }
                } catch (regexError) {
                    console.warn(`❌ Ошибка regex в парсере "${parser.name}":`, regexError.message);
                    continue;
                }
            }
        } catch (error) {
            console.error(`❌ Общая ошибка в парсере "${parser.name}":`, error);
        }
    }
    
    return foundId;
}

// Функция проверки валидности ID
function isValidId(id) {
    if (!id || typeof id !== 'string') return false;
    
    const cleanId = id.trim();
    
    // ОТСЕИВАЕМ МУСОРНЫЕ ЗНАЧЕНИЯ
    const invalidPatterns = [
        /^factfile$/i,
        /^testresult$/i, 
        /^partnersuggestions$/i,
        /^matches$/i,
        /^profile$/i,
        /^partner$/i,
        /^[a-z]+$/i, // Только буквы без цифр
        /^\d{1,3}$/,  // Слишком короткие числа
        /^[a-z]{1,5}$/i, // Слишком короткие буквенные ID
        /^https?:\/\//, // URL
        /^\/\//, // Неполные URL
        /^[.#]/, // CSS селекторы
        /^[\s\S]{1,3}$/ // Слишком короткие любые ID
    ];
    
    // Если попадает под мусорные паттерны - отсеиваем
    for (const pattern of invalidPatterns) {
        if (pattern.test(cleanId)) {
            console.log(`🚫 Отсеян мусорный ID: ${cleanId}`);
            return false;
        }
    }
    
    // ДОПУСТИМЫЕ ПАТТЕРНЫ ID
    const validPatterns = [
        /^[A-Z0-9]{8,12}$/, // Parship ID (PSPYD8M2)
        /^[a-zA-Z0-9_-]{6,30}$/, // Общие alphanumeric ID
        /^\d{4,20}$/, // Числовые ID (только длинные)
        /^[a-f0-9]{8}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{12}$/i, // UUID
        /^[a-zA-Z0-9]{15,35}$/ // Длинные ID (например, n1YR_jxwX3BzBg4erar39w2)
    ];
    
    // Должен соответствовать хотя бы одному валидному паттерну
    const isValid = validPatterns.some(pattern => pattern.test(cleanId));
    
    console.log(`🔍 Проверка ID "${cleanId}": ${isValid ? 'VALID' : 'INVALID'}`);
    return isValid;
}

// ==================== ПЕРЕСКАНИРОВАНИЕ СТРАНИЦЫ ====================

// Функция пересканирования страницы с новыми селекторами - ТОЛЬКО ДЛЯ НОВЫХ ЭЛЕМЕНТОВ
function rescanPageWithNewSelectors() {
    console.log('🔄 Пересканирование страницы с обновленными селекторами');
    
    // НЕ удаляем старые контролы! Только сбрасываем флаги для НОВЫХ элементов
    const currentSelectors = getCurrentSelectors();
    const allElements = document.querySelectorAll(currentSelectors);
    
    console.log(`🎯 Найдено элементов по новым селекторам: ${allElements.length}`);
    
    // Обрабатываем только элементы без флага enhanced
    allElements.forEach(el => {
        if (!el._universalEnhanced) {
            enhanceCard(el);
        }
    });
}

// "Тихое" обновление статуса без пересоздания контролов
async function quietUpdateCardStatuses() {
    const currentSelectors = getCurrentSelectors();
    const cards = document.querySelectorAll(currentSelectors);
    
    console.log(`🔍 Тихое обновление статусов: ${cards.length} карточек`);
    
    for (const el of cards) {
        if (el._universalEnhanced) {
            const id = extractIDFromElement(el);
            if (id) {
                await updateCardStatus(id, el);
            }
        }
    }
}


// ПАССИВНОЕ сканирование - только поиск без отправки
function scanPageWithParsers() {
    if (siteParsers.length === 0) {
        console.log("ℹ️ Нет активных парсеров для сканирования");
        return;
    }
    
    try {
        const pageHtml = document.documentElement.outerHTML;
        const foundIds = new Set();
        
        console.log(`🔍 Сканирование страницы с ${siteParsers.length} парсерами (только поиск)`);
        
        // Применяем парсеры для поиска ID по всей странице
        for (const parser of siteParsers) {
            try {
                if (parser.pattern && parser.pattern.length > 2) {
                    try {
                        const regex = new RegExp(parser.pattern, 'gi');
                        let match;
                        regex.lastIndex = 0;
                        
                        while ((match = regex.exec(pageHtml)) !== null) {
                            let extractedId = match[1] || match[0];
                            
                            if (extractedId && extractedId.length >= 3 && isValidId(extractedId)) {
                                extractedId = extractedId.trim();
                                console.log(`✅ Сканирование: парсер "${parser.name}" нашёл ID: ${extractedId}`);
                                foundIds.add(extractedId);
                            }
                        }
                    } catch (regexError) {
                        console.warn(`❌ Ошибка regex в сканировании (${parser.name}):`, regexError);
                    }
                }
            } catch (error) {
                console.error(`❌ Ошибка в парсере ${parser.name} при сканировании:`, error);
            }
        }
        
        // ТОЛЬКО ЛОГИРОВАНИЕ - НИКАКОЙ ОТПРАВКИ
        if (foundIds.size > 0) {
            console.log(`🎯 Сканирование найдено ${foundIds.size} ID (не отправляются автоматически):`, Array.from(foundIds));
            
            // Можно показать уведомление о найденных ID, но не отправлять
            showNotification(`🔍 Найдено ${foundIds.size} ID на странице`);
        }
        
    } catch (error) {
        console.error('❌ Ошибка сканирования:', error);
    }
}

// ==================== ЛОГИКА ПОДСВЕТКИ ====================

function shouldHighlight(records) {
    if (!Array.isArray(records) || records.length === 0) {
        return false;
    }
    
    const count = records.length;
    const useDup = siteSettings.useMinDuplicates;
    const useDate = siteSettings.useMinDate;

    if (!useDup && !useDate) {
        return true;
    }
    
    let meetsDuplicates = false;
    let meetsDate = false;

    if (useDup) {
        meetsDuplicates = count >= siteSettings.minDuplicates;
    }
    
    if (useDate) {
        const thresholdDate = new Date(siteSettings.minDate);
        meetsDate = records.some(r => {
            if (!r.created_at) return false;
            const recordDate = new Date(r.created_at);
            return !isNaN(recordDate) && recordDate >= thresholdDate;
        });
    }

    if (useDup && !useDate) {
        return meetsDuplicates;
    } else if (!useDup && useDate) {
        return meetsDate;
    } else if (useDup && useDate) {
        return meetsDuplicates || meetsDate;
    }

    return false;
}

function applyHighlight(element, shouldHighlight) {
    if (shouldHighlight) {
        element.style.transition = "background-color 0.3s ease";
        element.style.backgroundColor = "rgba(255,180,180,0.35)";
        element.style.outline = "2px solid #ff3366";
        element.style.outlineOffset = "1px";
        element.style.boxShadow = "0 0 10px rgba(255, 51, 102, 0.5)";
    } else {
        element.style.backgroundColor = "";
        element.style.outline = "";
        element.style.boxShadow = "";
    }
}

// ==================== УПРОЩЁННЫЕ API ЗАПРОСЫ ====================

async function getApiKey() {
    return new Promise((resolve) => {
        chrome.storage.local.get(["apiKey", "my_api_key"], (res) => {
            resolve(res.apiKey || res.my_api_key || null);
        });
    });
}

// ОБНОВЛЕННАЯ функция проверки записей с учетом групп
async function fetchRecordsForId(id) {
    const CACHE_TIME = 3000;
    const now = Date.now();
    
    // Проверяем кэш
    if (recordsCache.has(id)) {
        const cached = recordsCache.get(id);
        if (now - cached.timestamp < CACHE_TIME) {
            return cached.records;
        }
    }
    
    if (updateInProgress.has(id)) {
        return recordsCache.has(id) ? recordsCache.get(id).records : [];
    }
    
    updateInProgress.add(id);
    
    try {
        console.log(`🔍 Проверка ID ${id} с учетом групп...`);
        
        // Получаем настройки сайта (уже включают группы)
        const settingsResponse = await fetch(`${SERVER_URL}/api/site-settings?site=${encodeURIComponent(SITE_NAME)}`);
        const siteSettings = await settingsResponse.json();
        const siteGroups = siteSettings.siteGroups || [];
        
        console.log(`👥 Сайт ${SITE_NAME} принадлежит группам:`, siteGroups);
        
        // Собираем ВСЕ сайты для проверки: текущий сайт + все сайты из групп
        const sitesToCheck = new Set([SITE_NAME]);
        
        // Загружаем сайты из всех групп
        const groupPromises = siteGroups.map(async (groupName) => {
            try {
                console.log(`📋 Загрузка группы: ${groupName}`);
                const groupResponse = await fetch(`${SERVER_URL}/api/site-groups`);
                if (groupResponse.ok) {
                    const allGroups = await groupResponse.json();
                    const targetGroup = allGroups.find(g => g.group_name === groupName);
                    
                    if (targetGroup && targetGroup.site_list) {
                        console.log(`✅ Группа ${groupName} содержит сайты:`, targetGroup.site_list);
                        targetGroup.site_list.forEach(site => {
                            sitesToCheck.add(site);
                            console.log(`📍 Добавлен сайт для проверки: ${site}`);
                        });
                    }
                }
            } catch (groupError) {
                console.warn(`❌ Ошибка загрузки группы ${groupName}:`, groupError);
            }
        });
        
        // Ждем загрузки всех групп
        await Promise.all(groupPromises);
        
        console.log(`🌐 Всего сайтов для проверки ID ${id}:`, Array.from(sitesToCheck));
        
        // Параллельно проверяем ID на всех сайтах
        const checkPromises = Array.from(sitesToCheck).map(async (site) => {
            try {
                const url = `${SERVER_URL}/api/get-id/${encodeURIComponent(id)}?site=${encodeURIComponent(site)}`;
                console.log(`📡 Запрос к: ${url}`);
                
                const res = await fetch(url);
                if (res.ok) {
                    const data = await res.json();
                    console.log(`✅ Результат для ${site}:`, data.length, 'записей');
                    
                    // Добавляем информацию о сайте к каждой записи
                    return data.map(record => ({
                        ...record,
                        sourceSite: site, // Отмечаем откуда запись
                        isCurrentSite: site === SITE_NAME
                    }));
                } else {
                    console.warn(`❌ Ошибка запроса для ${site}:`, res.status);
                    return [];
                }
            } catch (siteError) {
                console.warn(`❌ Ошибка проверки на сайте ${site}:`, siteError);
                return [];
            }
        });
        
        // Ждем все результаты
        const results = await Promise.all(checkPromises);
        const allRecords = results.flat();
        
        console.log(`📊 ИТОГО для ID ${id}:`, {
            totalRecords: allRecords.length,
            fromCurrentSite: allRecords.filter(r => r.isCurrentSite).length,
            fromGroups: allRecords.filter(r => !r.isCurrentSite).length,
            sitesFound: [...new Set(allRecords.map(r => r.site || r.sourceSite))]
        });
        
        // Сохраняем в кэш
        recordsCache.set(id, {
            records: allRecords,
            timestamp: now,
            sitesChecked: Array.from(sitesToCheck)
        });
        
        return allRecords;
        
    } catch (error) {
        console.error("❌ Критическая ошибка загрузки записей:", error);
        return [];
    } finally {
        updateInProgress.delete(id);
    }
}

async function sendIdToServer(id, apiKey) {
    try {
        const res = await fetch(`${SERVER_URL}/api/add-id`, {
            method: "POST",
            headers: { "Content-Type": "application/json" },
            body: JSON.stringify({ id, apiKey, site: SITE_NAME })
        });
        return res.ok;
    } catch (e) {
        console.error("Ошибка отправки ID:", id, e);
        return false;
    }
}

// ==================== UI КОМПОНЕНТЫ ====================

function createControlButtons(id, initialRecords = []) {
    const container = document.createElement("div");
    container.className = "universal-id-controls";
    Object.assign(container.style, {
        position: "absolute",
        top: "4px",
        right: "4px",
        display: "flex",
        gap: "4px",
        zIndex: "99999",
        background: "rgba(255,255,255,0.9)",
        padding: "4px",
        borderRadius: "6px",
        border: "1px solid #ddd"
    });

    const copyBtn = document.createElement("button");
    copyBtn.textContent = "📋";
    copyBtn.title = `Копировать ID: ${id}`;
    Object.assign(copyBtn.style, {
        padding: "4px 8px",
        fontSize: "12px",
        cursor: "pointer",
        border: "1px solid #99ccee",
        borderRadius: "4px",
        backgroundColor: "#e0f0ff",
        color: "#003366"
    });
    
    copyBtn.addEventListener("click", async (e) => {
        e.preventDefault();
        e.stopPropagation();
        try {
            await navigator.clipboard.writeText(id);
            copyBtn.textContent = "✅";
            setTimeout(() => copyBtn.textContent = "📋", 1500);
        } catch {
            alert("Не удалось скопировать ID");
        }
    });

    const uploadBtn = document.createElement("button");
    uploadBtn.className = "universal-upload-btn";
    updateUploadButton(uploadBtn, initialRecords, id);

    uploadBtn.addEventListener("click", async (e) => {
        e.preventDefault();
        e.stopPropagation();
        
        if (uploadBtn._busy) return;
        uploadBtn._busy = true;
        
        uploadBtn.textContent = "⏳";
        uploadBtn.style.backgroundColor = "#fff3cd";
        uploadBtn.style.color = "#6a4f00";
        uploadBtn.disabled = true;

        const apiKey = await getApiKey();
        if (!apiKey) {
            alert("⚠️ API-ключ не задан. Кликните по иконке расширения и введите ключ.");
            uploadBtn._busy = false;
            updateUploadButton(uploadBtn, initialRecords, id);
            return;
        }

        const success = await sendIdToServer(id, apiKey);
        if (success) {
            recordsCache.delete(id);
            setTimeout(async () => {
                const records = await fetchRecordsForId(id);
                updateUploadButton(uploadBtn, records, id);
                applyHighlight(uploadBtn.closest(getCurrentSelectors()), shouldHighlight(records));
                uploadBtn._busy = false;
            }, 1000);
        } else {
            uploadBtn.textContent = "❌";
            setTimeout(async () => {
                const records = await fetchRecordsForId(id);
                updateUploadButton(uploadBtn, records, id);
                uploadBtn._busy = false;
            }, 1500);
        }
    });

    container.appendChild(copyBtn);
    container.appendChild(uploadBtn);
    return container;
}

function updateUploadButton(btn, records, id) {
    const hasRecords = Array.isArray(records) && records.length > 0;
    
    btn.textContent = hasRecords ? "☁️✔️" : "☁️";
    btn.disabled = false;
    
    Object.assign(btn.style, {
        padding: "4px 8px",
        fontSize: "12px",
        cursor: "pointer",
        border: hasRecords ? "1px solid #77cc88" : "1px solid #99cc99",
        borderRadius: "4px",
        backgroundColor: hasRecords ? "#2ecc71" : "#e9f9e9",
        color: hasRecords ? "white" : "#115511"
    });

    if (hasRecords) {
        // УЛУЧШЕННАЯ СТАТИСТИКА С УЧЕТОМ ГРУПП
        const recordsBySite = {};
        records.forEach(r => {
            const site = r.site || r.sourceSite || 'неизвестно';
            if (!recordsBySite[site]) {
                recordsBySite[site] = [];
            }
            recordsBySite[site].push(r);
        });

        let title = `ID: ${id}\n\n`;
        
        // Разделяем на текущий сайт и сайты из групп
        const currentSiteRecords = recordsBySite[SITE_NAME] || [];
        const groupSiteRecords = Object.entries(recordsBySite)
            .filter(([site]) => site !== SITE_NAME)
            .sort(([a], [b]) => a.localeCompare(b));
        
        // Текущий сайт
        if (currentSiteRecords.length > 0) {
            title += `📍 ТЕКУЩИЙ САЙТ (${SITE_NAME}): ${currentSiteRecords.length} записей\n`;
            currentSiteRecords.slice(0, 2).forEach(r => {
                const user = r.added_by || 'неизвестно';
                const time = new Date(r.created_at).toLocaleString('ru-RU', {
                    timeZone: 'Europe/Moscow',
                    day: "2-digit",
                    month: "2-digit", 
                    year: "numeric",
                    hour: "2-digit",
                    minute: "2-digit"
                });
                title += `   └ ${user} (${time})\n`;
            });
            if (currentSiteRecords.length > 2) {
                title += `   └ ... и ещё ${currentSiteRecords.length - 2} записей\n`;
            }
            title += '\n';
        }
        
        // Сайты из групп
        if (groupSiteRecords.length > 0) {
            title += `👥 НАЙДЕНО В ГРУППАХ:\n`;
            groupSiteRecords.forEach(([site, siteRecords]) => {
                title += `📍 ${site}: ${siteRecords.length} записей\n`;
                
                siteRecords.slice(0, 2).forEach(r => {
                    const user = r.added_by || 'неизвестно';
                    const time = new Date(r.created_at).toLocaleString('ru-RU', {
                        timeZone: 'Europe/Moscow',
                        day: "2-digit",
                        month: "2-digit", 
                        year: "numeric",
                        hour: "2-digit",
                        minute: "2-digit"
                    });
                    title += `   └ ${user} (${time})\n`;
                });
                
                if (siteRecords.length > 2) {
                    title += `   └ ... и ещё ${siteRecords.length - 2} записей\n`;
                }
                title += '\n';
            });
        }

        // ОБЩАЯ СТАТИСТИКА
        const totalCount = records.length;
        const currentSiteCount = currentSiteRecords.length;
        const groupsCount = totalCount - currentSiteCount;
        
        title += `📊 СВОДКА:\n`;
        title += `   Всего записей: ${totalCount}\n`;
        title += `   На этом сайте: ${currentSiteCount}\n`;
        title += `   В группах: ${groupsCount}\n`;
        title += `   Проверено сайтов: ${Object.keys(recordsBySite).length}\n\n`;
        title += `🖱️ Кликните, чтобы добавить снова`;

        btn.title = title;
        
        // Изменяем цвет кнопки если найдено в группах
        if (groupsCount > 0) {
            btn.style.backgroundColor = "#f39c12"; // оранжевый для групп
            btn.style.color = "white";
            btn.style.border = "1px solid #e67e22";
        }
    } else {
        btn.title = `Добавить ID ${id} на сервер`;
    }
}

// ==================== УВЕДОМЛЕНИЯ ====================

function showNotification(message) {
    const notification = document.createElement('div');
    notification.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        background: #10b981;
        color: white;
        padding: 12px 16px;
        border-radius: 8px;
        z-index: 10000;
        font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
        font-size: 14px;
        box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        max-width: 300px;
        word-wrap: break-word;
    `;
    
    notification.textContent = message;
    document.body.appendChild(notification);
    
    setTimeout(() => {
        if (notification.parentNode) {
            notification.parentNode.removeChild(notification);
        }
    }, 4000);
}

// ==================== ОСНОВНАЯ ЛОГИКА ====================

function enhanceCard(el) {
    if (!el || !el.nodeType || el.nodeType !== 1 || el._universalEnhanced) {
        return;
    }
    
    try {
        // ПРОВЕРЯЕМ СЕЛЕКТОРЫ ПАРСЕРОВ - если элемент не подходит ни под один селектор, пропускаем
        if (!elementMatchesAnyParserSelector(el)) {
            return;
        }
        
        const id = extractIDFromElement(el);
        if (!id) {
            return;
        }

        console.log(`✅ Найден ВАЛИДНЫЙ ID: ${id}`, el);
        el._universalEnhanced = true;
        el._lastEnhancedTime = Date.now(); // Добавляем временную метку
        
        // Устанавливаем позицию
        try {
            const style = window.getComputedStyle(el);
            if (style.position === "static") {
                el.style.position = "relative";
            }
        } catch (styleError) {
            // Игнорируем ошибки позиционирования
        }

        // Удаляем старые контролы ТОЛЬКО если они наши
        const oldControls = el.querySelector(".universal-id-controls");
        if (oldControls && oldControls._isUniversalControl) {
            oldControls.remove();
        }

        const controls = createControlButtons(id);
        controls._isUniversalControl = true; // Помечаем наши контролы
        el.appendChild(controls);

        // Обновляем статус
        updateCardStatus(id, el);
        
    } catch (error) {
        console.error('❌ Ошибка в enhanceCard:', error);
    }
}

async function updateCardStatus(id, element) {
    if (updateInProgress.has(id)) {
        return;
    }
    
    try {
        const records = await fetchRecordsForId(id);
        
        const btn = element.querySelector(".universal-upload-btn");
        if (btn) {
            updateUploadButton(btn, records, id);
        }
        
        applyHighlight(element, shouldHighlight(records));
    } catch (error) {
        console.warn("❌ Ошибка обновления статуса карточки:", error);
    }
}

// ==================== НАБЛЮДАТЕЛЬ И ИНИЦИАЛИЗАЦИЯ ====================

// Обновленный наблюдатель - с защитой от повторной обработки
const observer = new MutationObserver((mutations) => {
    mutations.forEach(mutation => {
        mutation.addedNodes.forEach(node => {
            if (node.nodeType === 1) { // Element node
                const currentSelectors = getCurrentSelectors();
                
                // Проверяем сам добавленный элемент
                if (node.matches && node.matches(currentSelectors) && !node._universalEnhanced) {
                    setTimeout(() => enhanceCard(node), 100); // Небольшая задержка
                }
                // Проверяем детей добавленного элемента
                if (node.querySelectorAll) {
                    node.querySelectorAll(currentSelectors).forEach(el => {
                        if (!el._universalEnhanced) {
                            setTimeout(() => enhanceCard(el), 100);
                        }
                    });
                }
            }
        });
    });
});

// Обновление каждые 15 секунд
setInterval(async () => {
    await loadSiteSettings();
    await loadSiteParsers(); // Теперь это также обновляет селекторы
    
    const currentSelectors = getCurrentSelectors();
    const cards = document.querySelectorAll(currentSelectors);
    cards.forEach(el => {
        const id = extractIDFromElement(el);
        if (id) updateCardStatus(id, el);
    });
    
}, 15000);

// Обновленная обработка SPA навигации
function handleUrlChange() {
    const newUrl = window.location.href;
    if (newUrl !== currentUrl) {
        currentUrl = newUrl;
        
        // Очищаем кэш при смене страницы
        recordsCache.clear();
        updateInProgress.clear();
        lastParsersLoad = 0;
        lastSettingsLoad = 0;
        
        // ПОЛНОЕ пересканирование только при смене страницы
        setTimeout(async () => {
            // Полностью сбрасываем все флаги
            document.querySelectorAll(getCurrentSelectors()).forEach(el => {
                delete el._universalEnhanced;
            });
            
            // Удаляем ВСЕ старые контролы
            document.querySelectorAll('.universal-id-controls').forEach(control => {
                if (control._isUniversalControl) {
                    control.remove();
                }
            });
            
            await loadSiteSettings();
            await loadSiteParsers();
            
            // Полное пересканирование
            const currentSelectors = getCurrentSelectors();
            const cards = document.querySelectorAll(currentSelectors);
            cards.forEach(enhanceCard);
            
            scanPageWithParsers();
        }, 2000);
    }
}

// Перехват SPA навигации
const originalPushState = history.pushState;
const originalReplaceState = history.replaceState;

history.pushState = function(...args) {
    originalPushState.apply(this, args);
    setTimeout(handleUrlChange, 100);
};

history.replaceState = function(...args) {
    originalReplaceState.apply(this, args);
    setTimeout(handleUrlChange, 100);
};

window.addEventListener('popstate', handleUrlChange);

// ==================== КОММУНИКАЦИЯ С POPUP ====================

chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    console.log('📨 Content script получил сообщение:', request);
    
    if (request.action === 'getParsersInfo') {
        sendResponse({
            success: true,
            parsers: siteParsers,
            site: SITE_NAME,
            settings: siteSettings,
            parsersCount: siteParsers.length,
            selectors: getCurrentSelectors(),
            selectorsCount: DYNAMIC_SELECTORS.split(', ').length
        });
        return true;
    }
    
    if (request.action === 'refreshParsers') {
        lastParsersLoad = 0;
        loadSiteParsers().then(() => {
            sendResponse({ 
                success: true,
                parsers: siteParsers,
                parsersCount: siteParsers.length,
                selectors: getCurrentSelectors()
            });
        });
        return true;
    }
    
    if (request.action === 'getSelectorsInfo') {
        sendResponse({
            selectors: getCurrentSelectors(),
            selectorsList: DYNAMIC_SELECTORS.split(', '),
            parsersWithSelectors: siteParsers.filter(p => p.selectors && p.selectors.length > 0)
        });
        return true;
    }
    
    if (request.action === 'rescanPage') {
        recordsCache.clear();
        rescanPageWithNewSelectors();
        sendResponse({ success: true });
        return true;
    }
    
    if (request.action === 'getSiteInfo') {
        sendResponse({
            site: SITE_NAME,
            settings: siteSettings
        });
        return true;
    }
    
    if (request.action === 'refreshSettings') {
        lastSettingsLoad = 0;
        lastParsersLoad = 0;
        recordsCache.clear();
        loadSiteSettings();
        loadSiteParsers();
        sendResponse({ success: true });
        return true;
    }
});

// ==================== ОТЛАДОЧНЫЕ ФУНКЦИИ ====================

// Функции для отладки селекторов
function debugSelectors() {
    console.log('🔍 ДЕБАГ СЕЛЕКТОРОВ:');
    console.log('Текущие селекторы:', getCurrentSelectors());
    console.log('Парсеры с селекторами:', siteParsers.filter(p => p.selectors && p.selectors.length > 0));
    
    const elements = document.querySelectorAll(getCurrentSelectors());
    console.log(`Найдено элементов: ${elements.length}`);
    
    elements.forEach((el, index) => {
        console.log(`Элемент ${index + 1}:`, el);
        console.log(`  HTML:`, el.outerHTML.substring(0, 200));
        
        // Проверяем какие селекторы совпадают
        const matchingSelectors = [];
        siteParsers.forEach(parser => {
            if (parser.selectors) {
                parser.selectors.forEach(selector => {
                    try {
                        if (el.matches(selector)) {
                            matchingSelectors.push(`${parser.name}: ${selector}`);
                        }
                    } catch (e) {
                        // Игнорируем ошибки селекторов
                    }
                });
            }
        });
        
        console.log(`  Совпадающие селекторы:`, matchingSelectors);
    });
}

// Сделаем доступным для отладки
window.debugSelectors = debugSelectors;

// ==================== ЗАПУСК ====================

// Запуск
(async function init() {
    console.log("🚀 Universal ID Collector запущен (СИСТЕМА СЕЛЕКТОРОВ)");
    await Promise.all([loadSiteSettings(), loadSiteParsers()]);
    observer.observe(document.body, { childList: true, subtree: true });
    
    // Обрабатываем существующие карточки
    const cards = document.querySelectorAll(getCurrentSelectors());
    cards.forEach(enhanceCard);
    
})();	